import 'dart:async';
import 'package:admin_desktop/domain/di/dependency_manager.dart';
import 'package:admin_desktop/infrastructure/services/app_connectivity.dart';
import 'package:admin_desktop/infrastructure/services/app_helpers.dart';
import 'package:admin_desktop/infrastructure/services/local_storage.dart';
import 'package:admin_desktop/infrastructure/services/tr_keys.dart';
import 'package:flutter/widgets.dart';
import 'package:geolocator/geolocator.dart';
import 'package:osm_nominatim/osm_nominatim.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';
import '../../domain/models/models.dart';
import 'select_address_state.dart';

class SelectAddressNotifier extends StateNotifier<SelectAddressState> {
  // final UsersRepository _usersRepository;
  Timer? _timer;
  final GeolocatorPlatform _geolocatorPlatform = GeolocatorPlatform.instance;

  SelectAddressNotifier()
      : super(SelectAddressState(textController: TextEditingController()));

  void setQuery(BuildContext context) {
    if (state.textController?.text.trim().isNotEmpty ?? false) {
      if (_timer?.isActive ?? false) {
        _timer?.cancel();
      }
      _timer = Timer(
        const Duration(milliseconds: 500),
        () {
          searchLocations();
        },
      );
    }
  }

  Future<void> searchLocations() async {
    state = state.copyWith(isSearching: true);
    try {
      final result = await Nominatim.searchByName(
        query: state.textController?.text.trim() ?? '',
        limit: 5,
        addressDetails: true,
        extraTags: true,
        nameDetails: true,
      );
      state = state.copyWith(searchedPlaces: result, isSearching: false);
    } catch (e) {
      debugPrint('===> search location error $e');
      state = state.copyWith(isSearching: false);
    }
  }

  void clearSearchField() {
    state.textController?.clear();
    state = state.copyWith(searchedPlaces: [], isSearching: false);
  }

  void setMapController(GoogleMapController controller) {
    state = state.copyWith(mapController: controller);
  }

  void setChoosing(bool value) {
    state = state.copyWith(isChoosing: value, isSearching: false);
  }

  void goLocationOpen({required Place place}) {
    state.textController?.text = place.displayName;
    state = state.copyWith(
        zipCode: place.address?['postcode'],
        location: LocationData(
            latitude: place.lat,
            longitude: place.lon,
            address: place.displayName),
        searchedPlaces: []);
  }

  void goToLocation({required Place place}) {
    state = state.copyWith(isSearching: false);
    state.textController?.text = place.displayName;
    state.mapController?.animateCamera(
      CameraUpdate.newCameraPosition(
        CameraPosition(
          bearing: 0,
          target: LatLng(place.lat, place.lon),
          tilt: 0,
          zoom: 17,
        ),
      ),
    );
    state = state.copyWith(
        zipCode: place.address?['postcode'],
        location: LocationData(
          latitude: place.lat,
          longitude: place.lon,
          address: place.displayName,
        ),
        searchedPlaces: []);
  }

  Future<void> gotToPlace(LocationData? location) async {
    state = state.copyWith(searchedPlaces: [], isSearching: false);
    Place? place;
    try {
      place = await Nominatim.reverseSearch(
        lat: location?.latitude,
        lon: location?.longitude,
        addressDetails: true,
        extraTags: true,
        nameDetails: true,
      );
      state.textController?.text = place.displayName;
      state = state.copyWith(
        zipCode: place.address?['postcode'],
        location: location?.copyWith(
          address: place.displayName,
        ),
      );
    } catch (e) {
      debugPrint('===> go to my location error: $e');
      state.textController?.text = '';
    }
    if (place != null) {
      state.mapController?.animateCamera(
        CameraUpdate.newCameraPosition(
          CameraPosition(
            bearing: 0,
            target: LatLng(place.lat, place.lon),
            tilt: 0,
            zoom: 17,
          ),
        ),
      );
      state = state.copyWith(
        location: LocationData(latitude: place.lat, longitude: place.lon),
      );
    }
  }

  Future<void> goToMyLocation(ValueChanged<LocationData> onchange) async {
    state = state.copyWith(searchingMyLocation: true);
    var check = await _geolocatorPlatform.checkPermission();
    LocationData? latLng;
    if (check == LocationPermission.denied ||
        check == LocationPermission.deniedForever) {
      check = await Geolocator.requestPermission();
      if (check != LocationPermission.denied &&
          check != LocationPermission.deniedForever) {
        var loc = await Geolocator.getCurrentPosition();
        latLng = LocationData(latitude: loc.latitude, longitude: loc.longitude);
        state.mapController!.animateCamera(CameraUpdate.newLatLngZoom(
            LatLng(loc.latitude, loc.longitude), 15));
      }
    } else {
      if (check != LocationPermission.deniedForever) {
        var loc = await Geolocator.getCurrentPosition();
        latLng = LocationData(latitude: loc.latitude, longitude: loc.longitude);
      }
    }
    state = state.copyWith(searchedPlaces: [], isSearching: false);
    Place? place;
    try {
      place = await Nominatim.reverseSearch(
        lat: latLng?.latitude,
        lon: latLng?.longitude,
        addressDetails: true,
        extraTags: true,
        nameDetails: true,
      );
      state.textController?.text = place.displayName;
    } catch (e) {
      debugPrint('===> go to my location error: $e');
      state.textController?.text = '';
      state = state.copyWith(searchingMyLocation: false);
    }
    if (place != null) {
      state.mapController?.animateCamera(
        CameraUpdate.newCameraPosition(
          CameraPosition(
            bearing: 0,
            target: LatLng(place.lat, place.lon),
            tilt: 0,
            zoom: 17,
          ),
        ),
      );
      onchange.call(LocationData(latitude: place.lat, longitude: place.lon));
      state = state.copyWith(
          location: LocationData(
            latitude: place.lat,
            longitude: place.lon,
          ),
          searchingMyLocation: false);
    }
  }

  Future<void> saveLocalAddress(
    bool? hasBack, {
    VoidCallback? onBack,
    VoidCallback? onGoMain,
  }) async {
    clearSearchField();
    state.mapController?.dispose();
  }

  Future<void> fetchLocationName(
    LocationData? latLng,
  ) async {
    state = state.copyWith(isButtonLoading: true);
    Place? place;
    try {
      place = await Nominatim.reverseSearch(
        lat: latLng?.latitude,
        lon: latLng?.longitude,
        addressDetails: true,
        extraTags: true,
        nameDetails: true,
      );
      state.textController?.text = place.displayName;
      state = state.copyWith(
        zipCode: place.address?['postcode'],
        location: latLng?.copyWith(
          address: place.displayName,
        ),
        isButtonLoading: false,
      );
    } catch (e) {
      state.textController?.text = '';
      state = state.copyWith(isButtonLoading: false);
    }
  }

  void setInitialLocation(LocationData? location) {
    state.textController?.text = location?.address ?? '';
    state = state.copyWith(location: location);
  }

  Future<void> checkDriverZone(
      {required BuildContext context,
      required LocationData? location,
      bool? isShopEdit}) async {
    if (isShopEdit ?? false) {
      state = state.copyWith(isActive: true);
      return;
    }
    final connected = await AppConnectivity.connectivity();
    if (connected) {
      state = state.copyWith(isLoading: true, isActive: false);
      final response = await usersRepository.checkDriverZone(
          location ?? LocationData(),
          LocalStorage.getUser()?.role == TrKeys.waiter
              ? LocalStorage.getUser()?.invite?.shopId ?? 0
              : LocalStorage.getUser()?.shop?.id ?? 0);
      response.when(
        success: (data) async {
          state = state.copyWith(isLoading: false, isActive: data);
          if (!data) {
            AppHelpers.showSnackBar(
              context,
              AppHelpers.getTranslation(TrKeys.noDriverZone),
            );
          }
        },
        failure: (failure, status) {
          state = state.copyWith(isLoading: false);
        },
      );
    } else {
      if (context.mounted) {
        AppHelpers.showSnackBar(
            context, AppHelpers.getTranslation(TrKeys.noInternetConnection));
      }
    }
  }
}
